<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Domain;
use App\Models\SecurityCheck;
use App\Models\SealAnalytics;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('verified');
    }

    public function index()
    {
        $user = Auth::user();
        
        // Get user statistics
        $stats = [
            'total_domains' => $user->domains()->count(),
            'verified_domains' => $user->verifiedDomains()->count(),
            'active_seals' => $user->domains()->whereHas('trustSeals', function($query) {
                $query->where('is_active', true);
            })->count(),
            'total_views' => $user->getTotalSealViews(),
            'avg_security_score' => round($user->getAverageSecurityScore(), 1)
        ];
        
        // Get recent domains
        $recentDomains = $user->domains()
            ->with(['trustSeals', 'securityChecks' => function($query) {
                $query->latest()->limit(1);
            }])
            ->latest()
            ->limit(5)
            ->get();
            
        // Get recent security alerts
        $securityAlerts = SecurityCheck::whereHas('domain', function($query) use ($user) {
                $query->where('user_id', $user->id);
            })
            ->where('status', 'failed')
            ->where('checked_at', '>=', now()->subDays(7))
            ->with('domain')
            ->latest()
            ->limit(5)
            ->get();
            
        // Get analytics data for the last 30 days
        $analyticsData = $this->getAnalyticsData($user);
        
        return view('dashboard.index', compact(
            'stats',
            'recentDomains', 
            'securityAlerts',
            'analyticsData'
        ));
    }
    
    private function getAnalyticsData($user)
    {
        $sealIds = $user->domains()
            ->with('trustSeals')
            ->get()
            ->pluck('trustSeals')
            ->flatten()
            ->pluck('id')
            ->toArray();
            
        if (empty($sealIds)) {
            return [
                'daily_views' => [],
                'top_countries' => [],
                'device_stats' => [],
                'browser_stats' => []
            ];
        }
        
        // Daily views for the last 30 days
        $dailyViews = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->toDateString();
            $views = SealAnalytics::whereIn('trust_seal_id', $sealIds)
                ->whereDate('viewed_at', $date)
                ->count();
            $dailyViews[] = [
                'date' => $date,
                'views' => $views
            ];
        }
        
        // Top countries
        $topCountries = SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->where('viewed_at', '>=', now()->subDays(30))
            ->selectRaw('country_code, COUNT(*) as views')
            ->whereNotNull('country_code')
            ->groupBy('country_code')
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();
            
        // Device statistics
        $deviceStats = SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->where('viewed_at', '>=', now()->subDays(30))
            ->selectRaw('device_type, COUNT(*) as views')
            ->whereNotNull('device_type')
            ->groupBy('device_type')
            ->orderBy('views', 'desc')
            ->get();
            
        // Browser statistics
        $browserStats = SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->where('viewed_at', '>=', now()->subDays(30))
            ->selectRaw('browser, COUNT(*) as views')
            ->whereNotNull('browser')
            ->groupBy('browser')
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();
        
        return [
            'daily_views' => $dailyViews,
            'top_countries' => $topCountries,
            'device_stats' => $deviceStats,
            'browser_stats' => $browserStats
        ];
    }
}