<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Subscription;
use Carbon\Carbon;

class SubscriptionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show subscription details and plans.
     */
    public function show()
    {
        $user = Auth::user();
        $currentSubscription = $user->currentSubscription;
        
        $plans = [
            'basic' => [
                'name' => 'Basic',
                'price' => 0,
                'domains' => 1,
                'features' => [
                    'Basic trust seal',
                    'Domain verification',
                    'Basic security monitoring',
                    'Email support'
                ]
            ],
            'professional' => [
                'name' => 'Professional',
                'price' => 29.99,
                'domains' => 10,
                'features' => [
                    'All Basic features',
                    'Custom trust seal design',
                    'Advanced security monitoring',
                    'Real-time alerts',
                    'Analytics dashboard',
                    'API access',
                    'Priority support'
                ]
            ],
            'enterprise' => [
                'name' => 'Enterprise',
                'price' => 99.99,
                'domains' => 100,
                'features' => [
                    'All Professional features',
                    'Unlimited domains',
                    'White-label solution',
                    'Advanced API features',
                    'Custom integrations',
                    'Dedicated support',
                    'SLA guarantee'
                ]
            ]
        ];
        
        return view('subscription.show', compact('currentSubscription', 'plans'));
    }

    /**
     * Subscribe to a plan.
     */
    public function subscribe(Request $request)
    {
        $request->validate([
            'plan' => ['required', 'in:professional,enterprise'],
            'billing_cycle' => ['required', 'in:monthly,yearly'],
            'payment_method' => ['required', 'string']
        ]);

        $user = Auth::user();
        
        // Check if user already has an active subscription
        if ($user->hasActiveSubscription()) {
            return redirect()->back()
                ->with('error', 'You already have an active subscription. Please cancel it first to change plans.');
        }

        $plan = $request->plan;
        $billingCycle = $request->billing_cycle;
        
        // Calculate pricing
        $prices = [
            'professional' => ['monthly' => 29.99, 'yearly' => 299.99],
            'enterprise' => ['monthly' => 99.99, 'yearly' => 999.99]
        ];
        
        $amount = $prices[$plan][$billingCycle];
        
        // Create subscription record
        $subscription = Subscription::create([
            'user_id' => $user->id,
            'plan' => $plan,
            'status' => 'pending',
            'amount' => $amount,
            'currency' => 'USD',
            'billing_cycle' => $billingCycle,
            'started_at' => now(),
            'expires_at' => $billingCycle === 'monthly' ? now()->addMonth() : now()->addYear(),
            'payment_method' => $request->payment_method,
            'payment_reference' => 'pending_' . uniqid(),
            'features' => $this->getPlanFeatures($plan)
        ]);

        // In a real application, you would integrate with a payment processor here
        // For demo purposes, we'll mark it as active immediately
        $subscription->update([
            'status' => 'active',
            'payment_reference' => 'demo_' . uniqid()
        ]);
        
        // Update user premium status
        $user->update([
            'is_premium' => true,
            'premium_expires_at' => $subscription->expires_at
        ]);

        return redirect()->route('subscription.show')
            ->with('success', 'Subscription activated successfully! Welcome to ' . ucfirst($plan) . ' plan.');
    }

    /**
     * Cancel subscription.
     */
    public function cancel(Request $request)
    {
        $user = Auth::user();
        $subscription = $user->currentSubscription;
        
        if (!$subscription || !$subscription->isActive()) {
            return redirect()->back()
                ->with('error', 'No active subscription found.');
        }

        $request->validate([
            'cancellation_reason' => ['nullable', 'string', 'max:500']
        ]);

        // Update subscription status
        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now()
        ]);
        
        // Keep premium features until expiration date
        // Premium status will be automatically updated by a scheduled job
        
        return redirect()->route('subscription.show')
            ->with('success', 'Subscription cancelled successfully. Your premium features will remain active until ' . 
                   $subscription->expires_at->format('M d, Y') . '.');
    }

    /**
     * Billing portal redirect.
     */
    public function billingPortal()
    {
        $user = Auth::user();
        
        if (!$user->hasActiveSubscription()) {
            return redirect()->route('subscription.show')
                ->with('error', 'No active subscription found.');
        }

        // In a real application, this would redirect to Stripe's billing portal
        // For demo purposes, redirect to subscription page
        return redirect()->route('subscription.show')
            ->with('info', 'Billing portal would be available with payment processor integration.');
    }

    /**
     * Handle Stripe webhook.
     */
    public function stripeWebhook(Request $request)
    {
        // In a real application, you would:
        // 1. Verify the webhook signature
        // 2. Handle different event types (payment_succeeded, payment_failed, etc.)
        // 3. Update subscription status accordingly
        
        $payload = $request->all();
        $eventType = $payload['type'] ?? null;
        
        switch ($eventType) {
            case 'invoice.payment_succeeded':
                $this->handlePaymentSucceeded($payload);
                break;
                
            case 'invoice.payment_failed':
                $this->handlePaymentFailed($payload);
                break;
                
            case 'customer.subscription.deleted':
                $this->handleSubscriptionCancelled($payload);
                break;
        }
        
        return response()->json(['status' => 'success']);
    }

    /**
     * Handle successful payment webhook.
     */
    private function handlePaymentSucceeded($payload)
    {
        // Extract customer ID and find subscription
        $customerId = $payload['data']['object']['customer'] ?? null;
        
        if (!$customerId) {
            return;
        }
        
        // Find subscription by payment reference or customer ID
        $subscription = Subscription::where('payment_reference', 'like', '%' . $customerId . '%')
            ->where('status', 'active')
            ->first();
            
        if ($subscription) {
            // Extend subscription period
            $newExpiryDate = $subscription->billing_cycle === 'monthly' 
                ? $subscription->expires_at->addMonth()
                : $subscription->expires_at->addYear();
                
            $subscription->update([
                'expires_at' => $newExpiryDate
            ]);
            
            // Update user premium status
            $subscription->user->update([
                'premium_expires_at' => $newExpiryDate
            ]);
        }
    }

    /**
     * Handle failed payment webhook.
     */
    private function handlePaymentFailed($payload)
    {
        $customerId = $payload['data']['object']['customer'] ?? null;
        
        if (!$customerId) {
            return;
        }
        
        $subscription = Subscription::where('payment_reference', 'like', '%' . $customerId . '%')
            ->where('status', 'active')
            ->first();
            
        if ($subscription) {
            // Mark subscription as past due
            $subscription->update([
                'status' => 'past_due'
            ]);
            
            // Send notification to user
            // In a real app, you would send an email notification here
        }
    }

    /**
     * Handle subscription cancellation webhook.
     */
    private function handleSubscriptionCancelled($payload)
    {
        $customerId = $payload['data']['object']['customer'] ?? null;
        
        if (!$customerId) {
            return;
        }
        
        $subscription = Subscription::where('payment_reference', 'like', '%' . $customerId . '%')
            ->whereIn('status', ['active', 'past_due'])
            ->first();
            
        if ($subscription) {
            $subscription->update([
                'status' => 'cancelled',
                'cancelled_at' => now()
            ]);
        }
    }

    /**
     * Get features for a plan.
     */
    private function getPlanFeatures($plan)
    {
        $features = [
            'professional' => [
                'domains' => 10,
                'custom_seal' => true,
                'advanced_monitoring' => true,
                'real_time_alerts' => true,
                'analytics' => true,
                'api_access' => true,
                'priority_support' => true
            ],
            'enterprise' => [
                'domains' => 100,
                'custom_seal' => true,
                'advanced_monitoring' => true,
                'real_time_alerts' => true,
                'analytics' => true,
                'api_access' => true,
                'priority_support' => true,
                'white_label' => true,
                'custom_integrations' => true,
                'dedicated_support' => true,
                'sla' => true
            ]
        ];
        
        return $features[$plan] ?? [];
    }

    /**
     * Upgrade subscription plan.
     */
    public function upgrade(Request $request)
    {
        $request->validate([
            'new_plan' => ['required', 'in:professional,enterprise']
        ]);

        $user = Auth::user();
        $currentSubscription = $user->currentSubscription;
        
        if (!$currentSubscription || !$currentSubscription->isActive()) {
            return redirect()->back()
                ->with('error', 'No active subscription found.');
        }

        $newPlan = $request->new_plan;
        
        // Prevent downgrade (in a real app, you might handle this differently)
        $planHierarchy = ['basic' => 0, 'professional' => 1, 'enterprise' => 2];
        if ($planHierarchy[$newPlan] <= $planHierarchy[$currentSubscription->plan]) {
            return redirect()->back()
                ->with('error', 'Cannot downgrade plan. Please contact support for assistance.');
        }

        // Calculate prorated amount (simplified)
        $prices = [
            'professional' => ['monthly' => 29.99, 'yearly' => 299.99],
            'enterprise' => ['monthly' => 99.99, 'yearly' => 999.99]
        ];
        
        $newAmount = $prices[$newPlan][$currentSubscription->billing_cycle];
        
        // Update subscription
        $currentSubscription->update([
            'plan' => $newPlan,
            'amount' => $newAmount,
            'features' => $this->getPlanFeatures($newPlan)
        ]);

        return redirect()->route('subscription.show')
            ->with('success', 'Subscription upgraded to ' . ucfirst($newPlan) . ' plan successfully!');
    }

    /**
     * Get subscription usage statistics.
     */
    public function usage()
    {
        $user = Auth::user();
        $subscription = $user->currentSubscription;
        
        if (!$subscription) {
            return redirect()->route('subscription.show');
        }

        $usage = [
            'domains_used' => $user->domains()->count(),
            'domains_limit' => $user->domainLimit,
            'seal_views_this_month' => $user->getTotalSealViews('month'),
            'api_calls_this_month' => 0, // Would be tracked separately
            'storage_used' => 0, // If applicable
        ];

        return view('subscription.usage', compact('subscription', 'usage'));
    }
}