<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use App\Models\Domain;

class StoreDomainRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth()->check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        $user = auth()->user();
        
        return [
            'domain' => [
                'required',
                'string',
                'max:255',
                'regex:/^(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\.)+(xn--[a-zA-Z0-9-]+|[a-zA-Z]{2,})$/',
                Rule::unique('domains')->where(function ($query) use ($user) {
                    return $query->where('user_id', $user->id);
                }),
                function ($attribute, $value, $fail) use ($user) {
                    // Check domain limit
                    if (!$user->canAddDomain()) {
                        $fail('You have reached your domain limit. Please upgrade your plan to add more domains.');
                    }
                    
                    // Validate domain format more strictly
                    if (!$this->isValidDomain($value)) {
                        $fail('The domain format is invalid.');
                    }
                    
                    // Check if domain is already taken by another user
                    if (Domain::where('domain', $value)->where('user_id', '!=', $user->id)->exists()) {
                        $fail('This domain is already registered by another user.');
                    }
                }
            ],
            'verification_method' => [
                'required',
                'string',
                Rule::in(['dns', 'file', 'meta'])
            ]
        ];
    }
    
    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'domain.required' => 'Please enter a domain name.',
            'domain.regex' => 'Please enter a valid domain name (e.g., example.com).',
            'domain.unique' => 'You have already added this domain.',
            'verification_method.required' => 'Please select a verification method.',
            'verification_method.in' => 'Please select a valid verification method.'
        ];
    }
    
    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        $domain = $this->input('domain');
        
        if ($domain) {
            // Clean and normalize domain
            $domain = strtolower(trim($domain));
            
            // Remove protocol if present
            $domain = preg_replace('/^https?:\/\//', '', $domain);
            
            // Remove www. prefix
            $domain = preg_replace('/^www\./', '', $domain);
            
            // Remove trailing slash
            $domain = rtrim($domain, '/');
            
            // Remove path if present
            $domain = explode('/', $domain)[0];
            
            $this->merge([
                'domain' => $domain
            ]);
        }
    }
    
    /**
     * Validate domain format more strictly.
     */
    protected function isValidDomain(string $domain): bool
    {
        // Basic format check
        if (!filter_var('http://' . $domain, FILTER_VALIDATE_URL)) {
            return false;
        }
        
        // Check for valid TLD
        $parts = explode('.', $domain);
        if (count($parts) < 2) {
            return false;
        }
        
        $tld = end($parts);
        if (strlen($tld) < 2 || !ctype_alpha($tld)) {
            return false;
        }
        
        // Check for reserved/invalid domains
        $reservedDomains = [
            'localhost',
            'test',
            'invalid',
            'example',
            'local'
        ];
        
        foreach ($reservedDomains as $reserved) {
            if (str_ends_with($domain, '.' . $reserved) || $domain === $reserved) {
                return false;
            }
        }
        
        return true;
    }
}