<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SealAnalytics extends Model
{
    use HasFactory;

    protected $fillable = [
        'trust_seal_id',
        'ip_address',
        'user_agent',
        'referrer',
        'country_code',
        'city',
        'browser',
        'os',
        'device_type',
        'viewed_at'
    ];

    protected $casts = [
        'viewed_at' => 'datetime',
    ];

    public function trustSeal(): BelongsTo
    {
        return $this->belongsTo(TrustSeal::class);
    }

    public function getDeviceTypeDisplayAttribute(): string
    {
        return match($this->device_type) {
            'mobile' => 'Mobile',
            'tablet' => 'Tablet',
            'desktop' => 'Desktop',
            default => 'Unknown'
        };
    }

    public function getBrowserDisplayAttribute(): string
    {
        return $this->browser ?: 'Unknown';
    }

    public function getOsDisplayAttribute(): string
    {
        return $this->os ?: 'Unknown';
    }

    public function getLocationDisplayAttribute(): string
    {
        $location = [];
        
        if ($this->city) {
            $location[] = $this->city;
        }
        
        if ($this->country_code) {
            $location[] = strtoupper($this->country_code);
        }
        
        return implode(', ', $location) ?: 'Unknown';
    }

    public function getReferrerDomainAttribute(): ?string
    {
        if (!$this->referrer) {
            return null;
        }
        
        $parsed = parse_url($this->referrer);
        return $parsed['host'] ?? null;
    }

    public function scopeToday($query)
    {
        return $query->whereDate('viewed_at', today());
    }

    public function scopeThisWeek($query)
    {
        return $query->whereBetween('viewed_at', [now()->startOfWeek(), now()->endOfWeek()]);
    }

    public function scopeThisMonth($query)
    {
        return $query->whereBetween('viewed_at', [now()->startOfMonth(), now()->endOfMonth()]);
    }

    public function scopeThisYear($query)
    {
        return $query->whereBetween('viewed_at', [now()->startOfYear(), now()->endOfYear()]);
    }

    public function scopeByCountry($query, $countryCode)
    {
        return $query->where('country_code', $countryCode);
    }

    public function scopeByDevice($query, $deviceType)
    {
        return $query->where('device_type', $deviceType);
    }

    public function scopeByBrowser($query, $browser)
    {
        return $query->where('browser', $browser);
    }

    public function scopeByReferrer($query, $referrer)
    {
        return $query->where('referrer', 'like', '%' . $referrer . '%');
    }

    public static function getTopCountries($trustSealId = null, $limit = 10)
    {
        $query = static::query()
            ->selectRaw('country_code, COUNT(*) as views')
            ->whereNotNull('country_code')
            ->groupBy('country_code')
            ->orderBy('views', 'desc')
            ->limit($limit);
            
        if ($trustSealId) {
            $query->where('trust_seal_id', $trustSealId);
        }
        
        return $query->get();
    }

    public static function getTopBrowsers($trustSealId = null, $limit = 10)
    {
        $query = static::query()
            ->selectRaw('browser, COUNT(*) as views')
            ->whereNotNull('browser')
            ->groupBy('browser')
            ->orderBy('views', 'desc')
            ->limit($limit);
            
        if ($trustSealId) {
            $query->where('trust_seal_id', $trustSealId);
        }
        
        return $query->get();
    }

    public static function getDeviceStats($trustSealId = null)
    {
        $query = static::query()
            ->selectRaw('device_type, COUNT(*) as views')
            ->whereNotNull('device_type')
            ->groupBy('device_type')
            ->orderBy('views', 'desc');
            
        if ($trustSealId) {
            $query->where('trust_seal_id', $trustSealId);
        }
        
        return $query->get();
    }
}