<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class User extends Authenticatable implements MustVerifyEmail
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone',
        'company',
        'website',
        'account_type',
        'is_premium',
        'premium_expires_at',
        'preferences',
        'email_notifications',
        'security_alerts',
        'last_login_at',
        'last_login_ip',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'is_premium' => 'boolean',
        'premium_expires_at' => 'datetime',
        'preferences' => 'array',
        'email_notifications' => 'boolean',
        'security_alerts' => 'boolean',
        'last_login_at' => 'datetime',
    ];

    public function domains(): HasMany
    {
        return $this->hasMany(Domain::class);
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    public function currentSubscription(): HasOne
    {
        return $this->hasOne(Subscription::class)
                    ->where('status', 'active')
                    ->where('expires_at', '>', now())
                    ->whereNull('cancelled_at')
                    ->latest();
    }

    public function verifiedDomains(): HasMany
    {
        return $this->domains()->where('status', 'verified');
    }

    public function activeDomains(): HasMany
    {
        return $this->domains()->where('is_active', true);
    }

    public function isPremium(): bool
    {
        return $this->is_premium && 
               ($this->premium_expires_at === null || $this->premium_expires_at > now());
    }

    public function hasActiveSubscription(): bool
    {
        return $this->currentSubscription !== null;
    }

    public function canAddDomain(): bool
    {
        if (!$this->isPremium()) {
            return $this->domains()->count() < 1;
        }
        
        $subscription = $this->currentSubscription;
        if (!$subscription) {
            return $this->domains()->count() < 1;
        }
        
        return $subscription->canAddDomain();
    }

    public function getDomainLimitAttribute(): int
    {
        if (!$this->isPremium()) {
            return 1;
        }
        
        $subscription = $this->currentSubscription;
        return $subscription ? $subscription->domain_limit : 1;
    }

    public function getAccountTypeDisplayAttribute(): string
    {
        return match($this->account_type) {
            'individual' => 'Individual',
            'business' => 'Business',
            'enterprise' => 'Enterprise',
            default => 'Individual'
        };
    }

    public function getDefaultPreferences(): array
    {
        return [
            'dashboard_theme' => 'light',
            'timezone' => 'UTC',
            'date_format' => 'Y-m-d',
            'notifications' => [
                'domain_verification' => true,
                'security_alerts' => true,
                'subscription_updates' => true,
                'weekly_reports' => false,
                'monthly_reports' => true
            ],
            'analytics' => [
                'show_visitor_details' => true,
                'track_referrers' => true,
                'geo_tracking' => true
            ]
        ];
    }

    public function getPreferencesAttribute($value): array
    {
        $default = $this->getDefaultPreferences();
        $custom = $value ? json_decode($value, true) : [];
        
        return array_merge_recursive($default, $custom);
    }

    public function updateLastLogin(string $ipAddress): void
    {
        $this->update([
            'last_login_at' => now(),
            'last_login_ip' => $ipAddress
        ]);
    }

    public function getTotalSealViews(): int
    {
        return $this->domains()
                   ->with('trustSeals')
                   ->get()
                   ->sum(function ($domain) {
                       return $domain->trustSeals->sum('display_count');
                   });
    }

    public function getAverageSecurityScore(): float
    {
        $domains = $this->verifiedDomains;
        
        if ($domains->isEmpty()) {
            return 0;
        }
        
        return $domains->avg('security_score') ?? 0;
    }
}
