<?php

namespace App\Notifications;

use App\Models\Domain;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\URL;

class DomainVerifiedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $domain;
    protected $verificationMethod;

    /**
     * Create a new notification instance.
     */
    public function __construct(Domain $domain, string $verificationMethod)
    {
        $this->domain = $domain;
        $this->verificationMethod = $verificationMethod;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        $channels = ['database'];
        
        // Check user preferences for email notifications
        $preferences = $notifiable->preferences ?? [];
        $notifications = $preferences['notifications'] ?? [];
        
        if ($notifications['email'] ?? true) {
            $channels[] = 'mail';
        }
        
        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $domainUrl = $this->getDomainUrl();
        $dashboardUrl = URL::route('dashboard');
        $domainManageUrl = URL::route('domains.show', $this->domain);
        
        return (new MailMessage)
            ->subject('Domain Verified Successfully - ' . $this->domain->domain)
            ->greeting('Great news!')
            ->line("Your domain **{$this->domain->domain}** has been successfully verified using the {$this->getVerificationMethodName()} method.")
            ->line('Your trust seal is now active and ready to display on your website.')
            ->action('View Domain Dashboard', $domainManageUrl)
            ->line('**What happens next:**')
            ->line('• Your trust seal has been automatically created and activated')
            ->line('• We\'ve started monitoring your domain\'s security')
            ->line('• You can now customize your trust seal appearance')
            ->line('• Analytics tracking has been enabled for your domain')
            ->line('')
            ->line('**Quick Setup:**')
            ->line('1. **Add the trust seal to your website** - Copy the embed code from your domain dashboard')
            ->line('2. **Customize the appearance** - Choose colors, size, and position that match your site')
            ->line('3. **Monitor your analytics** - Track how visitors interact with your trust seal')
            ->line('')
            ->action('Get Embed Code', $domainManageUrl)
            ->line('If you need help setting up your trust seal, our documentation has step-by-step guides.')
            ->line('Thank you for choosing our trust seal service!')
            ->salutation("Best regards,\nThe Trust Seal Team");
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'domain_verified',
            'title' => 'Domain Verified Successfully',
            'message' => "Your domain {$this->domain->domain} has been verified and your trust seal is now active.",
            'domain_id' => $this->domain->id,
            'domain' => $this->domain->domain,
            'verification_method' => $this->verificationMethod,
            'verified_at' => now()->toISOString(),
            'action_url' => URL::route('domains.show', $this->domain),
            'action_text' => 'View Domain',
            'icon' => 'check-circle',
            'color' => 'green',
            'priority' => 'normal',
            'metadata' => [
                'domain_id' => $this->domain->id,
                'verification_method' => $this->verificationMethod,
                'trust_seal_created' => true,
                'security_monitoring_started' => true
            ]
        ];
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'domain_id' => $this->domain->id,
            'domain' => $this->domain->domain,
            'verification_method' => $this->verificationMethod,
            'verified_at' => now()->toISOString(),
            'message' => "Domain {$this->domain->domain} has been successfully verified."
        ];
    }
    
    /**
     * Get human-readable verification method name.
     */
    protected function getVerificationMethodName(): string
    {
        return match($this->verificationMethod) {
            'dns' => 'DNS record',
            'file' => 'file upload',
            'meta' => 'HTML meta tag',
            default => $this->verificationMethod
        };
    }
    
    /**
     * Get the domain URL with protocol.
     */
    protected function getDomainUrl(): string
    {
        $domain = $this->domain->domain;
        
        // Add protocol if not present
        if (!str_starts_with($domain, 'http://') && !str_starts_with($domain, 'https://')) {
            $domain = 'https://' . $domain;
        }
        
        return $domain;
    }
    
    /**
     * Determine which queues should be used for which notification channels.
     */
    public function viaQueues(): array
    {
        return [
            'mail' => 'emails',
            'database' => 'default'
        ];
    }
    
    /**
     * Get the notification tags for organization.
     */
    public function tags(): array
    {
        return [
            'domain-verification',
            'user:' . $this->domain->user_id,
            'domain:' . $this->domain->id
        ];
    }
}