<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GeoLocationService
{
    protected $apiKey;
    protected $provider;
    protected $cacheTtl;

    public function __construct()
    {
        $this->apiKey = config('services.geolocation.api_key');
        $this->provider = config('services.geolocation.provider', 'ipapi');
        $this->cacheTtl = config('services.geolocation.cache_ttl', 3600); // 1 hour
    }

    /**
     * Get location data for an IP address.
     */
    public function getLocationByIp(string $ip): array
    {
        // Skip local/private IPs
        if ($this->isLocalIp($ip)) {
            return $this->getDefaultLocation();
        }

        $cacheKey = "geolocation:{$ip}";
        
        return Cache::remember($cacheKey, $this->cacheTtl, function () use ($ip) {
            try {
                return $this->fetchLocationData($ip);
            } catch (\Exception $e) {
                Log::warning('Geolocation service failed', [
                    'ip' => $ip,
                    'error' => $e->getMessage()
                ]);
                return $this->getDefaultLocation();
            }
        });
    }

    /**
     * Fetch location data from the selected provider.
     */
    protected function fetchLocationData(string $ip): array
    {
        return match($this->provider) {
            'ipapi' => $this->fetchFromIpApi($ip),
            'ipstack' => $this->fetchFromIpStack($ip),
            'ipgeolocation' => $this->fetchFromIpGeolocation($ip),
            default => $this->fetchFromIpApi($ip)
        };
    }

    /**
     * Fetch data from ip-api.com (free tier).
     */
    protected function fetchFromIpApi(string $ip): array
    {
        $response = Http::timeout(10)->get("http://ip-api.com/json/{$ip}", [
            'fields' => 'status,message,country,countryCode,region,regionName,city,lat,lon,timezone,isp,org,as'
        ]);

        if (!$response->successful()) {
            throw new \Exception('API request failed');
        }

        $data = $response->json();

        if ($data['status'] === 'fail') {
            throw new \Exception($data['message'] ?? 'Unknown error');
        }

        return [
            'ip' => $ip,
            'country' => $data['country'] ?? 'Unknown',
            'country_code' => $data['countryCode'] ?? 'XX',
            'region' => $data['regionName'] ?? 'Unknown',
            'region_code' => $data['region'] ?? 'XX',
            'city' => $data['city'] ?? 'Unknown',
            'latitude' => $data['lat'] ?? 0,
            'longitude' => $data['lon'] ?? 0,
            'timezone' => $data['timezone'] ?? 'UTC',
            'isp' => $data['isp'] ?? 'Unknown',
            'organization' => $data['org'] ?? 'Unknown',
            'as_number' => $data['as'] ?? 'Unknown',
            'provider' => 'ip-api'
        ];
    }

    /**
     * Fetch data from ipstack.com (requires API key).
     */
    protected function fetchFromIpStack(string $ip): array
    {
        if (!$this->apiKey) {
            throw new \Exception('IPStack API key not configured');
        }

        $response = Http::timeout(10)->get("http://api.ipstack.com/{$ip}", [
            'access_key' => $this->apiKey,
            'format' => 1
        ]);

        if (!$response->successful()) {
            throw new \Exception('API request failed');
        }

        $data = $response->json();

        if (isset($data['error'])) {
            throw new \Exception($data['error']['info'] ?? 'Unknown error');
        }

        return [
            'ip' => $ip,
            'country' => $data['country_name'] ?? 'Unknown',
            'country_code' => $data['country_code'] ?? 'XX',
            'region' => $data['region_name'] ?? 'Unknown',
            'region_code' => $data['region_code'] ?? 'XX',
            'city' => $data['city'] ?? 'Unknown',
            'latitude' => $data['latitude'] ?? 0,
            'longitude' => $data['longitude'] ?? 0,
            'timezone' => $data['time_zone']['id'] ?? 'UTC',
            'isp' => $data['connection']['isp'] ?? 'Unknown',
            'organization' => $data['connection']['isp'] ?? 'Unknown',
            'as_number' => 'Unknown',
            'provider' => 'ipstack'
        ];
    }

    /**
     * Fetch data from ipgeolocation.io (requires API key).
     */
    protected function fetchFromIpGeolocation(string $ip): array
    {
        if (!$this->apiKey) {
            throw new \Exception('IPGeolocation API key not configured');
        }

        $response = Http::timeout(10)->get('https://api.ipgeolocation.io/ipgeo', [
            'apiKey' => $this->apiKey,
            'ip' => $ip
        ]);

        if (!$response->successful()) {
            throw new \Exception('API request failed');
        }

        $data = $response->json();

        if (isset($data['message'])) {
            throw new \Exception($data['message']);
        }

        return [
            'ip' => $ip,
            'country' => $data['country_name'] ?? 'Unknown',
            'country_code' => $data['country_code2'] ?? 'XX',
            'region' => $data['state_prov'] ?? 'Unknown',
            'region_code' => $data['state_prov'] ?? 'XX',
            'city' => $data['city'] ?? 'Unknown',
            'latitude' => floatval($data['latitude'] ?? 0),
            'longitude' => floatval($data['longitude'] ?? 0),
            'timezone' => $data['time_zone']['name'] ?? 'UTC',
            'isp' => $data['isp'] ?? 'Unknown',
            'organization' => $data['organization'] ?? 'Unknown',
            'as_number' => 'AS' . ($data['asn'] ?? 'Unknown'),
            'provider' => 'ipgeolocation'
        ];
    }

    /**
     * Check if IP is local/private.
     */
    protected function isLocalIp(string $ip): bool
    {
        return !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
    }

    /**
     * Get default location data for local/unknown IPs.
     */
    protected function getDefaultLocation(): array
    {
        return [
            'ip' => 'unknown',
            'country' => 'Unknown',
            'country_code' => 'XX',
            'region' => 'Unknown',
            'region_code' => 'XX',
            'city' => 'Unknown',
            'latitude' => 0,
            'longitude' => 0,
            'timezone' => 'UTC',
            'isp' => 'Unknown',
            'organization' => 'Unknown',
            'as_number' => 'Unknown',
            'provider' => 'default'
        ];
    }

    /**
     * Get country information by country code.
     */
    public function getCountryInfo(string $countryCode): array
    {
        $countries = [
            'US' => ['name' => 'United States', 'continent' => 'North America'],
            'CA' => ['name' => 'Canada', 'continent' => 'North America'],
            'GB' => ['name' => 'United Kingdom', 'continent' => 'Europe'],
            'DE' => ['name' => 'Germany', 'continent' => 'Europe'],
            'FR' => ['name' => 'France', 'continent' => 'Europe'],
            'JP' => ['name' => 'Japan', 'continent' => 'Asia'],
            'CN' => ['name' => 'China', 'continent' => 'Asia'],
            'AU' => ['name' => 'Australia', 'continent' => 'Oceania'],
            'BR' => ['name' => 'Brazil', 'continent' => 'South America'],
            'IN' => ['name' => 'India', 'continent' => 'Asia'],
            // Add more countries as needed
        ];

        return $countries[strtoupper($countryCode)] ?? [
            'name' => 'Unknown',
            'continent' => 'Unknown'
        ];
    }

    /**
     * Calculate distance between two coordinates.
     */
    public function calculateDistance(float $lat1, float $lon1, float $lat2, float $lon2): float
    {
        $earthRadius = 6371; // km

        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);

        $a = sin($dLat/2) * sin($dLat/2) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * sin($dLon/2) * sin($dLon/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));

        return $earthRadius * $c;
    }

    /**
     * Get location statistics for analytics.
     */
    public function getLocationStats(array $ips): array
    {
        $stats = [
            'total_requests' => count($ips),
            'unique_ips' => count(array_unique($ips)),
            'countries' => [],
            'cities' => [],
            'top_countries' => [],
            'top_cities' => []
        ];

        foreach (array_unique($ips) as $ip) {
            $location = $this->getLocationByIp($ip);
            
            $country = $location['country'];
            $city = $location['city'];
            
            $stats['countries'][$country] = ($stats['countries'][$country] ?? 0) + 1;
            $stats['cities'][$city] = ($stats['cities'][$city] ?? 0) + 1;
        }

        // Sort and get top entries
        arsort($stats['countries']);
        arsort($stats['cities']);
        
        $stats['top_countries'] = array_slice($stats['countries'], 0, 10, true);
        $stats['top_cities'] = array_slice($stats['cities'], 0, 10, true);

        return $stats;
    }
}