<?php

namespace App\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class JsonAuthService
{
    private $dataPath;
    
    public function __construct()
    {
        $this->dataPath = storage_path('app/demo-data');
    }
    
    /**
     * Attempt to authenticate a user with email and password
     */
    public function attempt($email, $password)
    {
        $users = $this->getUsers();
        
        foreach ($users as $user) {
            if ($user['email'] === $email && Hash::check($password, $user['password'])) {
                // Store user session
                Session::put('auth_user', $user);
                Session::put('auth_user_id', $user['id']);
                Session::put('auth_user_role', $user['role']);
                
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get the currently authenticated user
     */
    public function user()
    {
        return Session::get('auth_user');
    }
    
    /**
     * Check if user is authenticated
     */
    public function check()
    {
        return Session::has('auth_user');
    }
    
    /**
     * Check if user is authenticated (alias for check)
     */
    public function isAuthenticated()
    {
        return $this->check();
    }
    
    /**
     * Check if user is admin
     */
    public function isAdmin()
    {
        $user = $this->user();
        return $user && $user['role'] === 'admin';
    }
    
    /**
     * Logout the user
     */
    public function logout()
    {
        Session::forget('auth_user');
        Session::forget('auth_user_id');
        Session::forget('auth_user_role');
    }
    
    /**
     * Get all users from JSON file
     */
    private function getUsers()
    {
        $filePath = $this->dataPath . '/users.json';
        
        if (!File::exists($filePath)) {
            return [];
        }
        
        $content = File::get($filePath);
        return json_decode($content, true) ?: [];
    }
    
    /**
     * Get demo domains
     */
    public function getDomains($userId = null)
    {
        $filePath = $this->dataPath . '/domains.json';
        
        if (!File::exists($filePath)) {
            return [];
        }
        
        $content = File::get($filePath);
        $domains = json_decode($content, true) ?: [];
        
        if ($userId) {
            return array_filter($domains, function($domain) use ($userId) {
                return $domain['user_id'] == $userId;
            });
        }
        
        return $domains;
    }
    
    /**
     * Get trust seals
     */
    public function getTrustSeals($domainId = null)
    {
        $filePath = $this->dataPath . '/trust_seals.json';
        
        if (!File::exists($filePath)) {
            return [];
        }
        
        $content = File::get($filePath);
        $seals = json_decode($content, true) ?: [];
        
        if ($domainId) {
            return array_filter($seals, function($seal) use ($domainId) {
                return $seal['domain_id'] == $domainId;
            });
        }
        
        return $seals;
    }
    
    /**
     * Get user statistics
     */
    public function getUserStats($userId)
    {
        $domains = $this->getDomains($userId);
        $trustSeals = $this->getTrustSeals($userId);
        
        return [
            'total_domains' => count($domains),
            'verified_domains' => count(array_filter($domains, function($d) { return $d['verification_status'] === 'verified'; })),
            'active_seals' => count($trustSeals),
            'total_views' => array_sum(array_column($trustSeals, 'views')),
            'security_score' => $this->calculateSecurityScore($domains)
        ];
    }
    
    /**
     * Calculate average security score for domains
     */
    private function calculateSecurityScore($domains)
    {
        if (empty($domains)) {
            return 0;
        }
        
        $totalScore = array_sum(array_column($domains, 'security_score'));
        return round($totalScore / count($domains));
    }
    
    /**
     * Add a new domain for demo purposes
     */
    public function addDomain($userId, $domainName)
    {
        $filePath = storage_path('app/demo-data/domains.json');
        $domains = [];
        
        if (file_exists($filePath)) {
            $domains = json_decode(file_get_contents($filePath), true) ?? [];
        }
        
        // Check if domain already exists
        foreach ($domains as $domain) {
            if ($domain['domain_name'] === $domainName) {
                return ['error' => 'Domain already exists'];
            }
        }
        
        // Create new domain
        $newDomain = [
            'id' => count($domains) + 1,
            'user_id' => $userId,
            'domain_name' => $domainName,
            'verification_status' => 'pending',
            'security_score' => rand(70, 95),
            'last_scan' => now()->toISOString(),
            'ssl_status' => 'valid',
            'ssl_expires_at' => now()->addDays(90)->toISOString(),
            'created_at' => now()->toISOString(),
            'updated_at' => now()->toISOString()
        ];
        
        $domains[] = $newDomain;
        
        // Save back to file
        file_put_contents($filePath, json_encode($domains, JSON_PRETTY_PRINT));
        
        return ['success' => true, 'domain' => $newDomain];
    }
}