<?php

namespace App\Services;

use App\Models\Domain;
use App\Models\SecurityCheck;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class SecurityMonitoringService
{
    /**
     * Run comprehensive security check for domain
     */
    public function runSecurityCheck(Domain $domain): array
    {
        $results = [];
        $overallScore = 0;
        $totalChecks = 0;
        
        try {
            // SSL Certificate Check
            $sslResult = $this->checkSSL($domain);
            $results['ssl'] = $sslResult;
            $overallScore += $sslResult['score'];
            $totalChecks++;
            
            // DNS Health Check
            $dnsResult = $this->checkDNS($domain);
            $results['dns'] = $dnsResult;
            $overallScore += $dnsResult['score'];
            $totalChecks++;
            
            // Safe Browsing Check
            $safeBrowsingResult = $this->checkSafeBrowsing($domain);
            $results['safe_browsing'] = $safeBrowsingResult;
            $overallScore += $safeBrowsingResult['score'];
            $totalChecks++;
            
            // Malware Check
            $malwareResult = $this->checkMalware($domain);
            $results['malware'] = $malwareResult;
            $overallScore += $malwareResult['score'];
            $totalChecks++;
            
            // Calculate final score
            $finalScore = $totalChecks > 0 ? round($overallScore / $totalChecks) : 0;
            
            // Store results in database
            $this->storeSecurityChecks($domain, $results, $finalScore);
            
            // Update domain security score
            $domain->update([
                'security_score' => $finalScore,
                'last_security_check' => now()
            ]);
            
            return [
                'success' => true,
                'score' => $finalScore,
                'results' => $results
            ];
            
        } catch (\Exception $e) {
            Log::error('Security check failed', [
                'domain' => $domain->domain,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'results' => $results
            ];
        }
    }
    
    /**
     * Check SSL certificate status
     */
    private function checkSSL(Domain $domain): array
    {
        try {
            $domainName = $domain->domain;
            $url = "https://{$domainName}";
            
            $context = stream_context_create([
                'ssl' => [
                    'capture_peer_cert' => true,
                    'verify_peer' => false,
                    'verify_peer_name' => false
                ]
            ]);
            
            $stream = @stream_socket_client(
                "ssl://{$domainName}:443",
                $errno,
                $errstr,
                30,
                STREAM_CLIENT_CONNECT,
                $context
            );
            
            if (!$stream) {
                return [
                    'status' => 'error',
                    'score' => 0,
                    'message' => 'SSL connection failed: ' . $errstr,
                    'details' => ['error' => $errstr, 'errno' => $errno]
                ];
            }
            
            $cert = stream_context_get_params($stream)['options']['ssl']['peer_certificate'];
            fclose($stream);
            
            if (!$cert) {
                return [
                    'status' => 'error',
                    'score' => 0,
                    'message' => 'No SSL certificate found',
                    'details' => []
                ];
            }
            
            $certData = openssl_x509_parse($cert);
            $expiryDate = Carbon::createFromTimestamp($certData['validTo_time_t']);
            $daysUntilExpiry = now()->diffInDays($expiryDate, false);
            
            // Calculate score based on expiry
            $score = 100;
            if ($daysUntilExpiry < 0) {
                $score = 0; // Expired
            } elseif ($daysUntilExpiry < 7) {
                $score = 30; // Expires soon
            } elseif ($daysUntilExpiry < 30) {
                $score = 70; // Expires within a month
            }
            
            $status = $daysUntilExpiry < 0 ? 'error' : ($daysUntilExpiry < 30 ? 'warning' : 'pass');
            
            // Update domain SSL info
            $domain->update([
                'ssl_info' => [
                    'valid' => $daysUntilExpiry > 0,
                    'expires_at' => $expiryDate->toDateTimeString(),
                    'days_until_expiry' => $daysUntilExpiry,
                    'issuer' => $certData['issuer']['CN'] ?? 'Unknown',
                    'subject' => $certData['subject']['CN'] ?? 'Unknown'
                ]
            ]);
            
            return [
                'status' => $status,
                'score' => $score,
                'message' => $this->getSSLMessage($daysUntilExpiry),
                'details' => [
                    'expires_at' => $expiryDate->toDateTimeString(),
                    'days_until_expiry' => $daysUntilExpiry,
                    'issuer' => $certData['issuer']['CN'] ?? 'Unknown'
                ]
            ];
            
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'score' => 0,
                'message' => 'SSL check failed: ' . $e->getMessage(),
                'details' => ['error' => $e->getMessage()]
            ];
        }
    }
    
    /**
     * Check DNS health
     */
    private function checkDNS(Domain $domain): array
    {
        try {
            $domainName = $domain->domain;
            $score = 100;
            $issues = [];
            
            // Check A record
            $aRecords = dns_get_record($domainName, DNS_A);
            if (empty($aRecords)) {
                $issues[] = 'No A record found';
                $score -= 30;
            }
            
            // Check MX record
            $mxRecords = dns_get_record($domainName, DNS_MX);
            if (empty($mxRecords)) {
                $issues[] = 'No MX record found';
                $score -= 20;
            }
            
            // Check NS records
            $nsRecords = dns_get_record($domainName, DNS_NS);
            if (empty($nsRecords)) {
                $issues[] = 'No NS records found';
                $score -= 25;
            }
            
            // Check for multiple NS records (redundancy)
            if (count($nsRecords) < 2) {
                $issues[] = 'Insufficient name server redundancy';
                $score -= 15;
            }
            
            $score = max(0, $score);
            $status = $score >= 80 ? 'pass' : ($score >= 50 ? 'warning' : 'error');
            
            return [
                'status' => $status,
                'score' => $score,
                'message' => empty($issues) ? 'DNS configuration looks good' : implode('; ', $issues),
                'details' => [
                    'a_records' => count($aRecords),
                    'mx_records' => count($mxRecords),
                    'ns_records' => count($nsRecords),
                    'issues' => $issues
                ]
            ];
            
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'score' => 0,
                'message' => 'DNS check failed: ' . $e->getMessage(),
                'details' => ['error' => $e->getMessage()]
            ];
        }
    }
    
    /**
     * Check Google Safe Browsing status
     */
    private function checkSafeBrowsing(Domain $domain): array
    {
        try {
            // Note: This is a placeholder. In production, you would use the actual Google Safe Browsing API
            // For demo purposes, we'll simulate the check
            $domainName = $domain->domain;
            
            // Simulate API call delay
            usleep(500000); // 0.5 second delay
            
            // For demo, assume most domains are safe unless they contain suspicious keywords
            $suspiciousKeywords = ['malware', 'phishing', 'spam', 'hack', 'virus'];
            $isSuspicious = false;
            
            foreach ($suspiciousKeywords as $keyword) {
                if (stripos($domainName, $keyword) !== false) {
                    $isSuspicious = true;
                    break;
                }
            }
            
            if ($isSuspicious) {
                return [
                    'status' => 'error',
                    'score' => 0,
                    'message' => 'Domain flagged as potentially unsafe',
                    'details' => ['threat_type' => 'suspicious_domain']
                ];
            } else {
                return [
                    'status' => 'pass',
                    'score' => 100,
                    'message' => 'No threats detected',
                    'details' => ['clean' => true]
                ];
            }
            
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'score' => 50, // Neutral score if check fails
                'message' => 'Safe browsing check failed: ' . $e->getMessage(),
                'details' => ['error' => $e->getMessage()]
            ];
        }
    }
    
    /**
     * Check for malware
     */
    private function checkMalware(Domain $domain): array
    {
        try {
            $domainName = $domain->domain;
            $url = "https://{$domainName}";
            
            // Attempt to fetch the homepage
            $response = Http::timeout(15)
                ->withOptions(['verify' => false])
                ->get($url);
                
            if (!$response->successful()) {
                return [
                    'status' => 'warning',
                    'score' => 70,
                    'message' => 'Unable to scan website content',
                    'details' => ['http_status' => $response->status()]
                ];
            }
            
            $content = $response->body();
            
            // Basic malware signature detection (simplified)
            $malwareSignatures = [
                'eval(base64_decode',
                'document.write(unescape',
                'iframe src="data:',
                '<script>eval(',
                'fromCharCode'
            ];
            
            $suspiciousContent = [];
            foreach ($malwareSignatures as $signature) {
                if (stripos($content, $signature) !== false) {
                    $suspiciousContent[] = $signature;
                }
            }
            
            if (!empty($suspiciousContent)) {
                return [
                    'status' => 'error',
                    'score' => 0,
                    'message' => 'Suspicious code patterns detected',
                    'details' => ['suspicious_patterns' => $suspiciousContent]
                ];
            }
            
            return [
                'status' => 'pass',
                'score' => 100,
                'message' => 'No malware signatures detected',
                'details' => ['clean' => true]
            ];
            
        } catch (\Exception $e) {
            return [
                'status' => 'warning',
                'score' => 70,
                'message' => 'Malware check incomplete: ' . $e->getMessage(),
                'details' => ['error' => $e->getMessage()]
            ];
        }
    }
    
    /**
     * Store security check results
     */
    private function storeSecurityChecks(Domain $domain, array $results, int $overallScore): void
    {
        foreach ($results as $checkType => $result) {
            SecurityCheck::create([
                'domain_id' => $domain->id,
                'check_type' => $checkType,
                'status' => $result['status'],
                'score' => $result['score'],
                'details' => $result['details'] ?? [],
                'message' => $result['message'],
                'checked_at' => now()
            ]);
        }
        
        // Store overall score check
        SecurityCheck::create([
            'domain_id' => $domain->id,
            'check_type' => 'overall',
            'status' => $overallScore >= 80 ? 'pass' : ($overallScore >= 50 ? 'warning' : 'error'),
            'score' => $overallScore,
            'details' => ['component_scores' => array_column($results, 'score', null)],
            'message' => "Overall security score: {$overallScore}/100",
            'checked_at' => now()
        ]);
    }
    
    /**
     * Get SSL status message
     */
    private function getSSLMessage(int $daysUntilExpiry): string
    {
        if ($daysUntilExpiry < 0) {
            return 'SSL certificate has expired';
        } elseif ($daysUntilExpiry < 7) {
            return "SSL certificate expires in {$daysUntilExpiry} days";
        } elseif ($daysUntilExpiry < 30) {
            return "SSL certificate expires in {$daysUntilExpiry} days";
        } else {
            return 'SSL certificate is valid';
        }
    }
    
    /**
     * Get domains that need security checks
     */
    public function getDomainsForSecurityCheck(): \Illuminate\Database\Eloquent\Collection
    {
        return Domain::where('is_active', true)
            ->where('status', 'verified')
            ->where(function($query) {
                $query->whereNull('last_security_check')
                    ->orWhere('last_security_check', '<', now()->subDay());
            })
            ->get();
    }
    
    /**
     * Run security checks for all eligible domains
     */
    public function runDailySecurityChecks(): array
    {
        $domains = $this->getDomainsForSecurityCheck();
        $results = [];
        
        foreach ($domains as $domain) {
            $results[$domain->domain] = $this->runSecurityCheck($domain);
            
            // Add small delay between checks to avoid overwhelming servers
            usleep(100000); // 0.1 second delay
        }
        
        return $results;
    }
    
    /**
     * Get security alerts for domain
     */
    public function getSecurityAlerts(Domain $domain): array
    {
        $alerts = [];
        
        // Check for expired SSL
        if ($domain->ssl_info && isset($domain->ssl_info['days_until_expiry'])) {
            $daysUntilExpiry = $domain->ssl_info['days_until_expiry'];
            if ($daysUntilExpiry < 0) {
                $alerts[] = [
                    'type' => 'error',
                    'title' => 'SSL Certificate Expired',
                    'message' => 'Your SSL certificate has expired and needs immediate attention.'
                ];
            } elseif ($daysUntilExpiry <= 7) {
                $alerts[] = [
                    'type' => 'warning',
                    'title' => 'SSL Certificate Expiring Soon',
                    'message' => "Your SSL certificate expires in {$daysUntilExpiry} days."
                ];
            }
        }
        
        // Check for low security score
        if ($domain->security_score < 50) {
            $alerts[] = [
                'type' => 'error',
                'title' => 'Low Security Score',
                'message' => "Your domain has a security score of {$domain->security_score}/100. This may affect your trust seal eligibility."
            ];
        } elseif ($domain->security_score < 80) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Security Score Needs Improvement',
                'message' => "Your domain has a security score of {$domain->security_score}/100. Consider addressing security issues."
            ];
        }
        
        // Check for failed security checks
        $failedChecks = $domain->securityChecks()
            ->where('status', 'error')
            ->where('checked_at', '>', now()->subDays(7))
            ->get();
            
        foreach ($failedChecks as $check) {
            $alerts[] = [
                'type' => 'error',
                'title' => 'Security Check Failed',
                'message' => $check->getDisplayName() . ': ' . $check->message
            ];
        }
        
        return $alerts;
    }
}