<?php

return [

    /*
    |--------------------------------------------------------------------------
    | Security Configuration
    |--------------------------------------------------------------------------
    |
    | This file contains configuration options for security monitoring,
    | domain verification, SSL checks, malware scanning, and threat detection.
    |
    */

    /*
    |--------------------------------------------------------------------------
    | General Settings
    |--------------------------------------------------------------------------
    */

    'enabled' => env('SECURITY_MONITORING_ENABLED', true),
    'debug' => env('SECURITY_DEBUG', false),
    'queue_connection' => env('SECURITY_QUEUE_CONNECTION', 'redis'),
    'queue_name' => env('SECURITY_QUEUE_NAME', 'security'),
    'timeout' => env('SECURITY_CHECK_TIMEOUT', 30), // seconds

    /*
    |--------------------------------------------------------------------------
    | Domain Verification
    |--------------------------------------------------------------------------
    */

    'verification' => [
        'methods' => [
            'dns' => [
                'enabled' => true,
                'record_type' => 'TXT',
                'record_name' => '_trustseal-verification',
                'ttl' => 300,
                'timeout' => 30,
                'retry_attempts' => 3,
                'retry_delay' => 5, // seconds
            ],
            'file' => [
                'enabled' => true,
                'filename' => 'trustseal-verification.txt',
                'path' => '/.well-known/',
                'timeout' => 15,
                'retry_attempts' => 3,
                'user_agent' => 'Trust Seal Portal Verification Bot/1.0',
                'follow_redirects' => true,
                'max_redirects' => 5,
            ],
            'meta' => [
                'enabled' => true,
                'tag_name' => 'trustseal-verification',
                'timeout' => 15,
                'retry_attempts' => 3,
                'user_agent' => 'Trust Seal Portal Verification Bot/1.0',
                'parse_html' => true,
            ],
        ],
        'token' => [
            'length' => 32,
            'characters' => 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789',
            'expires_after' => 24, // hours
        ],
        'auto_reverify' => [
            'enabled' => true,
            'interval_days' => 30,
            'max_failures' => 3,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | SSL Certificate Monitoring
    |--------------------------------------------------------------------------
    */

    'ssl' => [
        'enabled' => env('SSL_MONITORING_ENABLED', true),
        'check_frequency' => env('SSL_CHECK_FREQUENCY', 'daily'), // hourly, daily, weekly
        'timeout' => 30,
        'port' => 443,
        'verify_peer' => true,
        'verify_host' => true,
        'checks' => [
            'certificate_validity' => true,
            'certificate_chain' => true,
            'certificate_expiry' => true,
            'certificate_issuer' => true,
            'cipher_strength' => true,
            'protocol_version' => true,
            'mixed_content' => true,
        ],
        'alerts' => [
            'expiry_warning_days' => [30, 14, 7, 1],
            'chain_issues' => true,
            'weak_ciphers' => true,
            'outdated_protocols' => true,
        ],
        'trusted_cas' => [
            // List of trusted Certificate Authorities
            'Let\'s Encrypt',
            'DigiCert',
            'Comodo',
            'GeoTrust',
            'Thawte',
            'VeriSign',
            'GlobalSign',
            'Entrust',
        ],
        'minimum_key_size' => 2048,
        'allowed_protocols' => ['TLSv1.2', 'TLSv1.3'],
    ],

    /*
    |--------------------------------------------------------------------------
    | Malware Scanning
    |--------------------------------------------------------------------------
    */

    'malware' => [
        'enabled' => env('MALWARE_SCANNING_ENABLED', true),
        'providers' => [
            'google_safe_browsing' => [
                'enabled' => env('GOOGLE_SAFE_BROWSING_ENABLED', true),
                'api_key' => env('GOOGLE_SAFE_BROWSING_API_KEY'),
                'version' => 'v4',
                'threat_types' => [
                    'MALWARE',
                    'SOCIAL_ENGINEERING',
                    'UNWANTED_SOFTWARE',
                    'POTENTIALLY_HARMFUL_APPLICATION',
                ],
                'platform_types' => ['ANY_PLATFORM'],
                'threat_entry_types' => ['URL'],
                'timeout' => 10,
                'retry_attempts' => 2,
            ],
            'virustotal' => [
                'enabled' => env('VIRUSTOTAL_ENABLED', false),
                'api_key' => env('VIRUSTOTAL_API_KEY'),
                'timeout' => 15,
                'retry_attempts' => 2,
                'minimum_positives' => 2, // Minimum detections to consider malicious
            ],
            'urlvoid' => [
                'enabled' => env('URLVOID_ENABLED', false),
                'api_key' => env('URLVOID_API_KEY'),
                'timeout' => 10,
                'retry_attempts' => 2,
            ],
        ],
        'scan_frequency' => env('MALWARE_SCAN_FREQUENCY', 'daily'),
        'deep_scan' => [
            'enabled' => false, // Premium feature
            'check_subdomains' => true,
            'check_linked_resources' => true,
            'max_depth' => 2,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Blacklist Monitoring
    |--------------------------------------------------------------------------
    */

    'blacklist' => [
        'enabled' => env('BLACKLIST_MONITORING_ENABLED', true),
        'providers' => [
            'spamhaus' => [
                'enabled' => true,
                'zones' => [
                    'zen.spamhaus.org',
                    'sbl.spamhaus.org',
                    'css.spamhaus.org',
                ],
                'timeout' => 5,
            ],
            'surbl' => [
                'enabled' => true,
                'zones' => [
                    'multi.surbl.org',
                    'ws.surbl.org',
                    'ph.surbl.org',
                ],
                'timeout' => 5,
            ],
            'uribl' => [
                'enabled' => true,
                'zones' => [
                    'multi.uribl.com',
                    'black.uribl.com',
                    'red.uribl.com',
                ],
                'timeout' => 5,
            ],
        ],
        'check_frequency' => 'daily',
        'check_ip' => true,
        'check_domain' => true,
    ],

    /*
    |--------------------------------------------------------------------------
    | Website Health Monitoring
    |--------------------------------------------------------------------------
    */

    'health' => [
        'enabled' => env('HEALTH_MONITORING_ENABLED', true),
        'checks' => [
            'uptime' => [
                'enabled' => true,
                'frequency' => env('UPTIME_CHECK_FREQUENCY', 5), // minutes
                'timeout' => 10,
                'expected_status_codes' => [200, 301, 302],
                'user_agent' => 'Trust Seal Portal Health Monitor/1.0',
                'follow_redirects' => true,
                'max_redirects' => 5,
            ],
            'response_time' => [
                'enabled' => true,
                'warning_threshold' => 2000, // milliseconds
                'critical_threshold' => 5000, // milliseconds
            ],
            'content_check' => [
                'enabled' => false, // Premium feature
                'expected_text' => null,
                'unexpected_text' => ['error', 'maintenance', 'offline'],
                'check_trust_seal' => true,
            ],
        ],
        'notifications' => [
            'downtime' => [
                'enabled' => true,
                'channels' => ['mail', 'database'],
                'cooldown' => 300, // seconds
            ],
            'slow_response' => [
                'enabled' => true,
                'channels' => ['database'],
                'consecutive_failures' => 3,
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Security Headers Analysis
    |--------------------------------------------------------------------------
    */

    'headers' => [
        'enabled' => env('SECURITY_HEADERS_CHECK_ENABLED', true),
        'required_headers' => [
            'Strict-Transport-Security' => [
                'required' => true,
                'min_age' => 31536000, // 1 year
                'include_subdomains' => true,
            ],
            'X-Content-Type-Options' => [
                'required' => true,
                'expected_value' => 'nosniff',
            ],
            'X-Frame-Options' => [
                'required' => true,
                'expected_values' => ['DENY', 'SAMEORIGIN'],
            ],
            'X-XSS-Protection' => [
                'required' => false, // Deprecated but still checked
                'expected_value' => '1; mode=block',
            ],
            'Referrer-Policy' => [
                'required' => true,
                'expected_values' => [
                    'strict-origin-when-cross-origin',
                    'strict-origin',
                    'same-origin',
                ],
            ],
            'Content-Security-Policy' => [
                'required' => false, // Optional but recommended
                'check_directives' => true,
            ],
        ],
        'check_frequency' => 'weekly',
    ],

    /*
    |--------------------------------------------------------------------------
    | Vulnerability Scanning
    |--------------------------------------------------------------------------
    */

    'vulnerabilities' => [
        'enabled' => env('VULNERABILITY_SCANNING_ENABLED', false), // Premium feature
        'providers' => [
            'custom' => [
                'enabled' => true,
                'checks' => [
                    'exposed_files' => [
                        '.env',
                        '.git/config',
                        'wp-config.php',
                        'config.php',
                        'database.yml',
                    ],
                    'common_paths' => [
                        '/admin',
                        '/administrator',
                        '/wp-admin',
                        '/phpmyadmin',
                        '/cpanel',
                    ],
                    'server_info' => [
                        'check_server_header' => true,
                        'check_powered_by' => true,
                        'check_generator' => true,
                    ],
                ],
            ],
        ],
        'scan_frequency' => 'weekly',
    ],

    /*
    |--------------------------------------------------------------------------
    | Alert Thresholds
    |--------------------------------------------------------------------------
    */

    'alerts' => [
        'levels' => [
            'info' => [
                'color' => '#17a2b8',
                'icon' => 'info-circle',
                'notification_channels' => ['database'],
            ],
            'warning' => [
                'color' => '#ffc107',
                'icon' => 'exclamation-triangle',
                'notification_channels' => ['database', 'mail'],
            ],
            'error' => [
                'color' => '#dc3545',
                'icon' => 'exclamation-circle',
                'notification_channels' => ['database', 'mail'],
                'immediate' => true,
            ],
            'critical' => [
                'color' => '#721c24',
                'icon' => 'ban',
                'notification_channels' => ['database', 'mail', 'slack'],
                'immediate' => true,
                'escalate' => true,
            ],
        ],
        'rate_limiting' => [
            'enabled' => true,
            'max_per_hour' => 10,
            'cooldown_minutes' => 60,
        ],
        'escalation' => [
            'enabled' => env('ALERT_ESCALATION_ENABLED', false),
            'levels' => [
                'manager' => 30, // minutes
                'admin' => 60, // minutes
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Reporting
    |--------------------------------------------------------------------------
    */

    'reporting' => [
        'enabled' => true,
        'formats' => ['html', 'pdf', 'json'],
        'schedule' => [
            'daily' => [
                'enabled' => true,
                'time' => '09:00',
                'recipients' => 'user',
            ],
            'weekly' => [
                'enabled' => true,
                'day' => 'monday',
                'time' => '09:00',
                'recipients' => 'user',
            ],
            'monthly' => [
                'enabled' => false,
                'day' => 1,
                'time' => '09:00',
                'recipients' => 'user',
            ],
        ],
        'include_sections' => [
            'executive_summary' => true,
            'security_status' => true,
            'ssl_certificates' => true,
            'malware_scans' => true,
            'uptime_monitoring' => true,
            'security_headers' => true,
            'recommendations' => true,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Rate Limiting
    |--------------------------------------------------------------------------
    */

    'rate_limits' => [
        'security_checks' => [
            'per_minute' => 10,
            'per_hour' => 100,
            'per_day' => 1000,
        ],
        'domain_verification' => [
            'per_minute' => 5,
            'per_hour' => 20,
            'per_day' => 100,
        ],
        'api_requests' => [
            'per_minute' => 60,
            'per_hour' => 1000,
            'per_day' => 10000,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Caching
    |--------------------------------------------------------------------------
    */

    'cache' => [
        'enabled' => env('SECURITY_CACHE_ENABLED', true),
        'store' => env('SECURITY_CACHE_STORE', 'redis'),
        'prefix' => 'security:',
        'ttl' => [
            'domain_verification' => 3600, // 1 hour
            'ssl_check' => 14400, // 4 hours
            'malware_scan' => 7200, // 2 hours
            'blacklist_check' => 3600, // 1 hour
            'health_check' => 300, // 5 minutes
            'security_headers' => 21600, // 6 hours
            'vulnerability_scan' => 86400, // 24 hours
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | External Services
    |--------------------------------------------------------------------------
    */

    'external_services' => [
        'user_agents' => [
            'verification' => 'Trust Seal Portal Verification Bot/1.0 (+https://trustsealportal.com/bot)',
            'health_monitor' => 'Trust Seal Portal Health Monitor/1.0 (+https://trustsealportal.com/monitoring)',
            'security_scanner' => 'Trust Seal Portal Security Scanner/1.0 (+https://trustsealportal.com/security)',
        ],
        'timeouts' => [
            'dns_lookup' => 10,
            'http_request' => 30,
            'ssl_handshake' => 15,
        ],
        'retry_policy' => [
            'max_attempts' => 3,
            'delay_multiplier' => 2,
            'max_delay' => 60,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Security Scoring
    |--------------------------------------------------------------------------
    */

    'scoring' => [
        'enabled' => true,
        'weights' => [
            'ssl_certificate' => 25,
            'malware_free' => 30,
            'not_blacklisted' => 20,
            'uptime' => 15,
            'security_headers' => 10,
        ],
        'thresholds' => [
            'excellent' => 90,
            'good' => 75,
            'fair' => 60,
            'poor' => 40,
        ],
        'colors' => [
            'excellent' => '#28a745',
            'good' => '#6c757d',
            'fair' => '#ffc107',
            'poor' => '#dc3545',
        ],
    ],

];