<?php

return [

    /*
    |--------------------------------------------------------------------------
    | Webhook Configuration
    |--------------------------------------------------------------------------
    |
    | This file contains configuration options for handling webhooks from
    | various third-party services like payment providers, security services,
    | and analytics platforms.
    |
    */

    /*
    |--------------------------------------------------------------------------
    | General Settings
    |--------------------------------------------------------------------------
    */

    'enabled' => env('WEBHOOKS_ENABLED', true),
    'debug' => env('WEBHOOKS_DEBUG', false),
    'queue_connection' => env('WEBHOOKS_QUEUE_CONNECTION', 'redis'),
    'queue_name' => env('WEBHOOKS_QUEUE_NAME', 'webhooks'),
    'timeout' => env('WEBHOOK_TIMEOUT', 30), // seconds
    'retry_attempts' => env('WEBHOOK_RETRY_ATTEMPTS', 3),
    'retry_delay' => env('WEBHOOK_RETRY_DELAY', 5), // seconds

    /*
    |--------------------------------------------------------------------------
    | Security Settings
    |--------------------------------------------------------------------------
    */

    'security' => [
        'verify_signatures' => env('WEBHOOK_VERIFY_SIGNATURES', true),
        'allowed_ips' => [
            // Stripe IPs
            '54.187.174.169',
            '54.187.205.235',
            '54.187.216.72',
            '54.241.31.99',
            '54.241.31.102',
            '54.241.34.107',
            // PayPal IPs
            '173.0.81.1/25',
            '173.0.81.33/32',
            '173.0.82.1/25',
            '173.0.82.33/32',
            // Add other service IPs as needed
        ],
        'rate_limiting' => [
            'enabled' => true,
            'max_attempts' => 100,
            'decay_minutes' => 1,
        ],
        'signature_tolerance' => 300, // seconds
    ],

    /*
    |--------------------------------------------------------------------------
    | Stripe Webhooks
    |--------------------------------------------------------------------------
    */

    'stripe' => [
        'enabled' => env('STRIPE_WEBHOOKS_ENABLED', true),
        'endpoint_secret' => env('STRIPE_WEBHOOK_SECRET'),
        'events' => [
            // Payment Events
            'payment_intent.succeeded' => [
                'handler' => 'App\Services\Webhooks\Stripe\PaymentSucceededHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'payment_intent.payment_failed' => [
                'handler' => 'App\Services\Webhooks\Stripe\PaymentFailedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'payment_method.attached' => [
                'handler' => 'App\Services\Webhooks\Stripe\PaymentMethodAttachedHandler',
                'queue' => 'default',
                'enabled' => true,
            ],

            // Subscription Events
            'customer.subscription.created' => [
                'handler' => 'App\Services\Webhooks\Stripe\SubscriptionCreatedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'customer.subscription.updated' => [
                'handler' => 'App\Services\Webhooks\Stripe\SubscriptionUpdatedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'customer.subscription.deleted' => [
                'handler' => 'App\Services\Webhooks\Stripe\SubscriptionDeletedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'customer.subscription.trial_will_end' => [
                'handler' => 'App\Services\Webhooks\Stripe\TrialWillEndHandler',
                'queue' => 'default',
                'enabled' => true,
            ],

            // Invoice Events
            'invoice.payment_succeeded' => [
                'handler' => 'App\Services\Webhooks\Stripe\InvoicePaymentSucceededHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'invoice.payment_failed' => [
                'handler' => 'App\Services\Webhooks\Stripe\InvoicePaymentFailedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'invoice.upcoming' => [
                'handler' => 'App\Services\Webhooks\Stripe\UpcomingInvoiceHandler',
                'queue' => 'default',
                'enabled' => true,
            ],

            // Customer Events
            'customer.created' => [
                'handler' => 'App\Services\Webhooks\Stripe\CustomerCreatedHandler',
                'queue' => 'default',
                'enabled' => true,
            ],
            'customer.updated' => [
                'handler' => 'App\Services\Webhooks\Stripe\CustomerUpdatedHandler',
                'queue' => 'default',
                'enabled' => true,
            ],
            'customer.deleted' => [
                'handler' => 'App\Services\Webhooks\Stripe\CustomerDeletedHandler',
                'queue' => 'default',
                'enabled' => true,
            ],

            // Dispute Events
            'charge.dispute.created' => [
                'handler' => 'App\Services\Webhooks\Stripe\DisputeCreatedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
        ],
        'signature_header' => 'Stripe-Signature',
        'tolerance' => 300, // seconds
    ],

    /*
    |--------------------------------------------------------------------------
    | PayPal Webhooks
    |--------------------------------------------------------------------------
    */

    'paypal' => [
        'enabled' => env('PAYPAL_WEBHOOKS_ENABLED', false),
        'webhook_id' => env('PAYPAL_WEBHOOK_ID'),
        'events' => [
            'PAYMENT.SALE.COMPLETED' => [
                'handler' => 'App\Services\Webhooks\PayPal\PaymentCompletedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'PAYMENT.SALE.DENIED' => [
                'handler' => 'App\Services\Webhooks\PayPal\PaymentDeniedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'BILLING.SUBSCRIPTION.CREATED' => [
                'handler' => 'App\Services\Webhooks\PayPal\SubscriptionCreatedHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
            'BILLING.SUBSCRIPTION.CANCELLED' => [
                'handler' => 'App\Services\Webhooks\PayPal\SubscriptionCancelledHandler',
                'queue' => 'high',
                'enabled' => true,
            ],
        ],
        'verify_ssl' => true,
    ],

    /*
    |--------------------------------------------------------------------------
    | Security Service Webhooks
    |--------------------------------------------------------------------------
    */

    'security' => [
        'enabled' => env('SECURITY_WEBHOOKS_ENABLED', true),
        'providers' => [
            'google_safe_browsing' => [
                'enabled' => env('GSB_WEBHOOKS_ENABLED', false),
                'secret' => env('GSB_WEBHOOK_SECRET'),
                'events' => [
                    'threat_detected' => [
                        'handler' => 'App\Services\Webhooks\Security\ThreatDetectedHandler',
                        'queue' => 'urgent',
                        'enabled' => true,
                    ],
                    'threat_resolved' => [
                        'handler' => 'App\Services\Webhooks\Security\ThreatResolvedHandler',
                        'queue' => 'high',
                        'enabled' => true,
                    ],
                ],
            ],
            'virustotal' => [
                'enabled' => env('VIRUSTOTAL_WEBHOOKS_ENABLED', false),
                'secret' => env('VIRUSTOTAL_WEBHOOK_SECRET'),
                'events' => [
                    'scan_completed' => [
                        'handler' => 'App\Services\Webhooks\Security\ScanCompletedHandler',
                        'queue' => 'high',
                        'enabled' => true,
                    ],
                ],
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Analytics Webhooks
    |--------------------------------------------------------------------------
    */

    'analytics' => [
        'enabled' => env('ANALYTICS_WEBHOOKS_ENABLED', true),
        'providers' => [
            'google_analytics' => [
                'enabled' => env('GA_WEBHOOKS_ENABLED', false),
                'secret' => env('GA_WEBHOOK_SECRET'),
                'events' => [
                    'goal_completed' => [
                        'handler' => 'App\Services\Webhooks\Analytics\GoalCompletedHandler',
                        'queue' => 'default',
                        'enabled' => true,
                    ],
                    'anomaly_detected' => [
                        'handler' => 'App\Services\Webhooks\Analytics\AnomalyDetectedHandler',
                        'queue' => 'default',
                        'enabled' => true,
                    ],
                ],
            ],
            'mixpanel' => [
                'enabled' => env('MIXPANEL_WEBHOOKS_ENABLED', false),
                'secret' => env('MIXPANEL_WEBHOOK_SECRET'),
                'events' => [
                    'funnel_completed' => [
                        'handler' => 'App\Services\Webhooks\Analytics\FunnelCompletedHandler',
                        'queue' => 'default',
                        'enabled' => true,
                    ],
                ],
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Custom Webhooks
    |--------------------------------------------------------------------------
    */

    'custom' => [
        'enabled' => env('CUSTOM_WEBHOOKS_ENABLED', true),
        'endpoints' => [
            'domain_verification' => [
                'handler' => 'App\Services\Webhooks\Custom\DomainVerificationHandler',
                'queue' => 'high',
                'enabled' => true,
                'secret' => env('DOMAIN_WEBHOOK_SECRET'),
            ],
            'ssl_expiry' => [
                'handler' => 'App\Services\Webhooks\Custom\SslExpiryHandler',
                'queue' => 'default',
                'enabled' => true,
                'secret' => env('SSL_WEBHOOK_SECRET'),
            ],
            'uptime_monitor' => [
                'handler' => 'App\Services\Webhooks\Custom\UptimeMonitorHandler',
                'queue' => 'default',
                'enabled' => true,
                'secret' => env('UPTIME_WEBHOOK_SECRET'),
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Logging
    |--------------------------------------------------------------------------
    */

    'logging' => [
        'enabled' => env('WEBHOOK_LOGGING_ENABLED', true),
        'level' => env('WEBHOOK_LOG_LEVEL', 'info'),
        'channel' => env('WEBHOOK_LOG_CHANNEL', 'webhooks'),
        'log_requests' => env('LOG_WEBHOOK_REQUESTS', true),
        'log_responses' => env('LOG_WEBHOOK_RESPONSES', true),
        'log_failures' => env('LOG_WEBHOOK_FAILURES', true),
        'retention_days' => env('WEBHOOK_LOG_RETENTION_DAYS', 30),
    ],

    /*
    |--------------------------------------------------------------------------
    | Monitoring
    |--------------------------------------------------------------------------
    */

    'monitoring' => [
        'enabled' => env('WEBHOOK_MONITORING_ENABLED', true),
        'metrics' => [
            'track_processing_time' => true,
            'track_success_rate' => true,
            'track_failure_rate' => true,
            'track_retry_attempts' => true,
        ],
        'alerts' => [
            'failure_threshold' => 5, // consecutive failures
            'processing_time_threshold' => 30, // seconds
            'success_rate_threshold' => 95, // percentage
        ],
        'health_check' => [
            'enabled' => true,
            'endpoint' => '/webhooks/health',
            'interval' => 60, // seconds
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Notification Settings
    |--------------------------------------------------------------------------
    */

    'notifications' => [
        'enabled' => env('WEBHOOK_NOTIFICATIONS_ENABLED', true),
        'channels' => [
            'webhook_failure' => ['mail', 'slack'],
            'webhook_success' => ['database'],
            'webhook_retry' => ['database'],
        ],
        'recipients' => [
            'admin' => env('WEBHOOK_ADMIN_EMAIL', 'admin@example.com'),
            'developer' => env('WEBHOOK_DEVELOPER_EMAIL', 'dev@example.com'),
        ],
        'cooldown' => [
            'failure_notifications' => 300, // seconds
            'success_notifications' => 0, // no cooldown
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Queue Configuration
    |--------------------------------------------------------------------------
    */

    'queues' => [
        'urgent' => [
            'connection' => 'redis',
            'queue' => 'urgent',
            'timeout' => 60,
            'retry_after' => 90,
            'max_tries' => 5,
        ],
        'high' => [
            'connection' => 'redis',
            'queue' => 'high',
            'timeout' => 120,
            'retry_after' => 180,
            'max_tries' => 3,
        ],
        'default' => [
            'connection' => 'redis',
            'queue' => 'default',
            'timeout' => 180,
            'retry_after' => 240,
            'max_tries' => 3,
        ],
        'low' => [
            'connection' => 'redis',
            'queue' => 'low',
            'timeout' => 300,
            'retry_after' => 360,
            'max_tries' => 2,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Validation Rules
    |--------------------------------------------------------------------------
    */

    'validation' => [
        'enabled' => env('WEBHOOK_VALIDATION_ENABLED', true),
        'rules' => [
            'max_payload_size' => 1024 * 1024, // 1MB
            'required_headers' => [
                'User-Agent',
                'Content-Type',
            ],
            'allowed_content_types' => [
                'application/json',
                'application/x-www-form-urlencoded',
            ],
            'timeout' => 30, // seconds
        ],
        'sanitization' => [
            'strip_tags' => true,
            'trim_whitespace' => true,
            'decode_html_entities' => true,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Error Handling
    |--------------------------------------------------------------------------
    */

    'error_handling' => [
        'log_errors' => true,
        'notify_on_error' => true,
        'retry_failed_webhooks' => true,
        'dead_letter_queue' => [
            'enabled' => true,
            'queue' => 'webhook-failures',
            'retention_days' => 7,
        ],
        'circuit_breaker' => [
            'enabled' => true,
            'failure_threshold' => 10,
            'timeout' => 300, // seconds
            'recovery_timeout' => 600, // seconds
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Testing
    |--------------------------------------------------------------------------
    */

    'testing' => [
        'enabled' => env('WEBHOOK_TESTING_ENABLED', false),
        'mock_responses' => [
            'stripe' => [
                'payment_intent.succeeded' => [
                    'id' => 'pi_test_123',
                    'status' => 'succeeded',
                    'amount' => 2000,
                    'currency' => 'usd',
                ],
            ],
        ],
        'test_endpoints' => [
            'stripe' => '/webhooks/test/stripe',
            'paypal' => '/webhooks/test/paypal',
            'security' => '/webhooks/test/security',
            'analytics' => '/webhooks/test/analytics',
        ],
    ],

];